/*
** Copyright (c) 2005
**	Jeff Forys (jeffware@marjum.com).  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that: (1) Redistributions of
** source code must retain the above copyright notice, this list of
** conditions and the following disclaimer, (2) Redistributions in
** binary form must reproduce the above copyright notice, this list
** of conditions and the following disclaimer in the documentation
** and/or other materials provided with the distribution, (3) All
** advertising materials mentioning features or use of this software
** must display the following acknowledgment: ``This product includes
** software developed by Jeff Forys (jeffware@marjum.com).'', (4)
** The name of the author may not be used to endorse or promote products
** derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/

const char *Version = "1.4 20090916";

const char *CopyrightVersion = "axmjpeg %s\n\nCopyright (c) 2005 Jeff Forys.  All rights reserved.\n\nRedistribution and use in source and binary forms, with or without\nmodification, is permitted under the terms specified in the source\ncode (and manual pages).\n";

#ifndef lint
static char rcsid[] = "$Id: axmjpeg.c,v 1.14 2010/09/27 01:31:10 forys Exp $";
#endif

#define	_SOCKADDR_LEN		/* Tru64 Unix */
#define	_USE_IRS		/* IBM AIX */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <sysexits.h>
#include <stdint.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>

/*
 * Portability #define's...
 */
#ifdef	__alpha			/* some vendors (e.g., Sun) define BSD empty */
#if defined(BSD) && (BSD == 198911)	/* assume Compaq Tru64 */
#define	NO_HSTRERROR
#endif
#endif

#if defined(__hpux) || defined(sun)
#define	NO_HSTRERROR
#define	NO_SINLEN
#endif

#if defined(__linux)
#define	NO_SINLEN
#endif

#ifndef	INADDR_NONE	/* SunOS */
#define	INADDR_NONE	0xffffffff 
#endif

#ifndef	MIN		/* SunOS */
#define	MIN(a,b)	(( (a) < (b) )? (a): (b))
#endif

#define	IMGBUFSIZ	(1024*1024)
#define	HDRBUFSIZ	512

typedef	int		sysex_t;

/* command name and options */
char	*progname;
char	*opt_httphost, *opt_httpport, *opt_httppath, *opt_httpauth;
char	*opt_cmd[256], opt_auth[256];
char	*opt_outfil, *opt_renfil;
int	opt_numfil = 0, opt_nummax = 0, opt_numext = -1;
int	opt_debug = 0, opt_loband = 0, opt_skpcnt = 0;
int	opt_binhdr_gp = 0, opt_binhdr_tn = 0;

/* image buffer operations */
unsigned char imgbuf[IMGBUFSIZ];
size_t	imglen;
volatile int ReadImage = 1;

/* prototypes */
int	Usage(const char *);
sysex_t	GetImage(int);
void	PutImage(const char *, const char *, const char *);
void	ImageTimer(int);
ssize_t	filbuf(int, void *, size_t);
int	GetSock(const char *, const char *);
sysex_t	SendURL(int, const char *, const char *);
void	base64encode(const void *, size_t, char *, size_t);

/*
 * axmjpeg - parse a Video mjpeg stream that follows Revision 2.x of
 * the "Axis Video API, HTTP Interface Specification, documented here:
 *
 *    http://www.axis.com/techsup/cam_servers/dev/cam_http_api.htm
 *
 * This program also works with cameras that follow "Revision 1.x" of
 * this specification provided the "Content-Length:" header is enabled.
 *
 * This program also works with the TRENDnet Internet Camera Server
 * (http://www.trendnet.com), specifically tested with the TV-IP110.
 * To use a TRENDnet camera (or any other camera that prepends a
 * header to each JPEG image), specify the '-T' flag.
 *
 * More specifically, this program wants to find an <image> of size
 * <imagelen> in a stream formated as:
 *
 *      <any>\n
 *	Content-Length:[<space>]<imagelen>[<any>][\r]\n
 *	[\r]\n
 *      <image>
 *
 * Where <space> is any whitespace, <imagelen> consists of digits, and
 * <any> is anything (including headers followed by [\r]\n sequences).
 *
 *
 *
 * Axmjpeg can also read images from the SQ IP Cam and GP-280 IP Camera
 * as they use a method of streaming that is similar to the Axis cameras.
 * Here, each image in the http stream is preceeded by a 40-byte binary:
 *
 *	<header><image><header><image> ...
 *
 * Encoded in the header is the size of the image.  To read images from
 * these cameras, the "-N" flag must be specified to axmjpeg.
 *
 * Support for SQ IP Cam, GP-280, and possibly other cameras that
 * send images prepended with a 40-byte binary header courtesy of
 * Sam Liddicott <sam@liddicott.com>.
 */

/*
 * SQ IP Cam, GP-280, binary header format.
 *
 * The only fields we use are 'bom' and 'imglen'.
 *
 * Note: These 16-bit short integers arrive in little-endian,
 * as opposed to network byte order (which is big-endian).
 */
typedef struct {
	uint16_t junk[10];	/* (all zero) */
	uint16_t bom;		/* GP_HDR_BOM */
	uint16_t imglen;	/* size of <image> in bytes */
	uint16_t j1;		/* (zero) */
	uint16_t width;		/* width of image */
	uint16_t height;	/* height of image */
	uint16_t junk2[5];      /* junk2[0]=0x46 (rest are zeros) */
} binhdr_gp_t;
#define	GP_HDRSIZE	sizeof(binhdr_gp_t)
#define	GP_HDR_BOM	0x2a2a

/*
 * TRENDnet Internet Camera Servers prepend a header to each
 * JPEG image.  The one I have prepends a 28-byte header, so
 * "-x 28" would work.  The '-T' option looks for a JPEG SOI
 * marker in the image.
 */
#define	TN_HDR_SOI1	0xFF
#define	TN_HDR_SOI2	0xD8

/*
 *
 * Each image, or optionally images at some specified interval may
 * be written to a (optionally unique) file and/or passed to one or
 * more external applications for further processing.
 *
 * While axmjpeg can read the stream piped into stdin (e.g., using
 * the "wget" tool) it can also open a connection to an HTTP server
 * on it's own (thereby eliminating the pipe and context switches).
 */
int
main(int argc, char *argv[])
{
	struct sigaction act;
	sigset_t iomask, mask;
	struct itimerval it_s;
	int infd, cmdidx, i, ch, capival = 0, conival = -1;
	sysex_t exstat;

	/* set progname to path-stripped argv[0] */
	if ((progname = strrchr(argv[0], '/')) != NULL)
		progname++;
	else
		progname = argv[0];

	/* parse args */
	for (ch = 0; ch < sizeof(opt_cmd)/sizeof(*opt_cmd); ch++)
		opt_cmd[ch] = 0;

	cmdidx = 0;
	while ((ch = getopt(argc, argv, "c:de:i:I:m:nNo:r:R:Tvx:")) != -1) {
		switch (ch) {
		    case 'c':
			if (cmdidx < sizeof(opt_cmd)/sizeof(*opt_cmd))
				opt_cmd[cmdidx++] = optarg;
			else
				return Usage("too many commands");
			break;
		    case 'd':
			opt_debug++;
			break;
		    case 'e':
			if (!isdigit(*optarg))
				return Usage("invalid exit file count (-e)");
			opt_numext = atoi(optarg);
			break;
		    case 'I':
			opt_loband++;
		    case 'i':
			if (!isdigit(*optarg))
				return Usage("invalid capture interval (-i/I)");
			capival = atoi(optarg);
			if (capival == 0)
				return Usage("zero capture interval (-i/I)");
			break;
		    case 'm':
			if (!isdigit(*optarg))
				return Usage("invalid max file count (-m)");
			opt_nummax = atoi(optarg);
			break;
		    case 'n':
			opt_numfil++;
			break;
		    case 'N':
			opt_binhdr_gp++;
			break;
		    case 'T':
			opt_binhdr_tn++;
			break;
		    case 'o':
			if (opt_outfil != NULL)
				return Usage("too many output files");
			opt_outfil = optarg;
			break;
		    case 'r':
			if (opt_renfil != NULL)
				return Usage("too many rename files");
			opt_renfil = optarg;
			break;
		    case 'R':
			if (!isdigit(*optarg))
				return Usage("invalid reconnect interval (-R)");
			conival = atoi(optarg);
			break;
		    case 'v':
			printf(CopyrightVersion, Version);
			return EX_OK;
		    case 'x':
			if (!isdigit(*optarg))
				return Usage("invalid header skip count (-x)");
			opt_skpcnt = atoi(optarg);
			break;
		    default:
			return Usage("unknown flag");
		}
	}
	argc -= optind;
	argv += optind;

	if (argc > 1)
		return Usage("multiple URLs");

	/*
	 * Simple http URL parser; chop an http URL into 'opt_httphost',
	 * 'opt_httpport' (default "80") and 'opt_httppath'.
	 *
	 * Also permit "basic access" authorization as in:
	 *	http://user:pass@Example.COM/cgi/mjpg/mjpeg.cgi
	 * where "user:pass" will be encoded to base-64 in 'opt_httpauth'.
	 */
	if (argc == 1 && argv[0][0] == '-' && argv[0][1] == '\0') {
		argc = 0;		/* URL of "-" implies STDIN */
	} else if (argc == 1) {
		ch = 0;
		if (strncmp(argv[0], "http://", 7) != 0)
			return Usage("non-HTTP URL");
		else {
			char *cp;

			opt_httpauth = NULL;
			opt_httphost = argv[0] + 7;
			if ((cp = strchr(opt_httphost,'/')) == NULL)
				ch = 1;
			else {
				*cp++ = '\0';
				opt_httppath = cp;
				if ((cp = strchr(opt_httphost,'@')) != NULL) {
					opt_httpauth = opt_httphost;
					opt_httphost= cp + 1;
					*cp = '\0';
					base64encode(opt_httpauth,
						strlen(opt_httpauth), opt_auth,
						sizeof(opt_auth) /
							sizeof(*opt_auth));
					opt_auth[sizeof(opt_auth) /
						sizeof(*opt_auth) - 1] = '\0';
					opt_httpauth = opt_auth;
				}
				if ((cp = strchr(opt_httphost,':')) != NULL) {
					*cp++ = '\0';
					opt_httpport = cp;
					while (isdigit(*cp))
						cp++;
					if (cp == opt_httpport || *cp != '\0')
						ch = 1;
				}
				if (*opt_httppath == '\0')
					ch = 1;
			}
		}
		if (ch == 1)
			return Usage("invalid URL");
		if (*opt_httphost == '\0')
			opt_httphost = "127.0.0.1";
		if (opt_httpport == NULL)
			opt_httpport = "80";
	}

	/*
	 * Command line restrictions:
	 *
	 * -I (low-bandwidth): requires a URL.
	 * -R (reconnect-on-error): requires a URL.
	 * -I and -R: these options are mutually exclusive.
	 */
	if (opt_loband && (capival == 0 || argc == 0))
		return Usage("'-I' requested but URL not");
	if (conival >= 0 && argc == 0)
		return Usage("'-R' requested but URL not");
	if (opt_loband && conival >= 0)
		return Usage("mutually exclusive '-I' and '-R' options");

	if (opt_debug)
		fprintf(stderr, "%s: args parsed, debug enabled (%d)\n",
			progname, opt_debug);

	/*
	 * Configure signal handlers:
	 *	SIGHUP -> ignored.
	 *	SIGALRM -> invokes ImageTimer().
	 */
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = SIG_IGN;
	sigaction(SIGHUP, &act, (struct sigaction *)NULL);
	sigaddset(&act.sa_mask, SIGALRM);
	act.sa_handler = ImageTimer;
	sigaction(SIGALRM, &act, (struct sigaction *)NULL);

	/* SIGALRM blocked while doing I/O to avoid EINTR's */
	sigemptyset(&iomask);
	sigaddset(&iomask, SIGALRM);

	if (argc == 1) {
		int retry;
		do {
			retry = 0;
			if ((infd = GetSock(opt_httphost, opt_httpport)) < 0) {
				if (conival >= 0)
					retry = 1;
				else
					return EX_NOHOST;
			}
			if (!retry && (exstat = SendURL(infd, opt_httppath,
						opt_httpauth)) != EX_OK) {
				if (conival >= 0)
					retry = 1;
				else
					return exstat;
			}
			if (retry) {
				if (opt_debug)
				    fprintf(stderr,
					"%s: reconnecting in %d seconds...\n",
					progname, conival);
				if (conival > 0)
					sleep(conival);
			}
		} while (retry == 1);
	} else
		infd = STDIN_FILENO;

	/*
	 * If a capture interval was specified, configure itimer.
	 */
	if (capival != 0) {
		it_s.it_value.tv_sec = it_s.it_interval.tv_sec =
			capival / 1000;
		it_s.it_value.tv_usec = it_s.it_interval.tv_usec =
			(capival % 1000) * 1000;
		setitimer(ITIMER_REAL, &it_s, NULL);
	}

	/*
	 * Loop reading images (parsing the image data stream) and
	 * writing them out to files / piping them to commands.
	 */
	for (;;) {
		sigprocmask(SIG_BLOCK, &iomask, &mask);
		exstat = GetImage(infd);
		sigprocmask(SIG_UNBLOCK, &iomask, NULL);

		/*
		 * If the image read was successful and we've tripped
		 * over the capture interval or there is no interval,
		 * process the image.
		 *
		 * If the image read failed and we arent doing loband
		 * or reconnects, break out of the loop with the error.
		 */
		if (exstat == EX_OK) {
			if (ReadImage || capival == 0) {
				/*
				 * Catch case where opt_numext initially 0.
				 * This permits user to do sanity check on
				 * exit status without writing any data.
				 */
				if (opt_numext-- == 0)
					break;

				sigprocmask(SIG_BLOCK, &iomask, &mask);
				ReadImage = 0;
				PutImage(NULL, opt_outfil, opt_renfil);
				for (i = 0; i < cmdidx; i++)
					PutImage(opt_cmd[i], NULL, NULL);
				sigprocmask(SIG_UNBLOCK, &iomask, NULL);

				if (opt_numext == 0)
					break;	/* reached max images out */
			}
		} else if (exstat == EX_SOFTWARE || (!opt_loband && conival<0))
			break;			/* exit on error */

		/*
		 * For low bandwidth connections and connection errors:
		 *	1) Disconnect the socket,
		 *	2) In the "reconnect on error" case sleep()
		 *	   the number of seconds specified by -R,
		 *	3) Wait for next interval timer to go off,
		 *	4) Reconnect and request a new image.
		 */
		if (opt_loband || exstat != EX_OK) {
			sigprocmask(SIG_BLOCK, &iomask, &mask);
			for (;;) {
				if (infd >= 0)
					close(infd);

				if (opt_debug && conival >= 0)
					fprintf(stderr, "%s: reconnecting in %d seconds...\n", progname, conival);
				if (conival > 0) {
					sigprocmask(SIG_UNBLOCK, &iomask, NULL);
					sleep(conival);
					sigprocmask(SIG_BLOCK, &iomask, &mask);
				}

				if (capival > 0)
					while (!ReadImage)
						sigsuspend(&mask);

				if ((infd = GetSock(opt_httphost,
				                    opt_httpport)) < 0 ||
				            SendURL(infd, opt_httppath,
				                    opt_httpauth) != EX_OK)
					ReadImage = 0;	/* try again... */
				else
					break;
			}
			sigprocmask(SIG_UNBLOCK, &iomask, NULL);
		}
	}

	/* return EX_OK on EOF, or else whatever is in 'exstat' */
	return exstat == EX_NOINPUT? EX_OK: exstat;
}

int
Usage(const char *msg)
{
	char *sp, spcbuf[128];
	int len;

	if ((len = strlen(progname)) > sizeof(spcbuf))
		len = sizeof(spcbuf) - 1;
	for (sp = spcbuf; len > 0; sp++, len--)
		*sp = ' ';
	*sp = '\0';

	fprintf(stderr, "%s: invalid argument: %s specified\n", progname, msg);
	fprintf(stderr,
	  "usage: %s [-dnvNT] [ -i <msecs> | [-I <msecs> | -R <secs>] ]\n",
		progname);
	fprintf(stderr,
	  "       %s [-o <outfil>] [-r <renfil>] [-m <maxfil>] [-e <exitcnt>]\n",
		 spcbuf);
	fprintf(stderr,
	  "       %s [-x <skipcnt>]  { [-c <cmd>] }  [<URL>]\n",
		spcbuf);
	return EX_USAGE;
}

/*
 * ImageTimer(sig)
 *
 * ITIMER periodically signals us to request a new image at the specified
 * interval; just set 'ReadImage = 1'.
 */
void
ImageTimer(int sig)
{
	ReadImage = 1;
}

/*
 * GetImage(fd)
 *
 * Read image from 'fd' into 'imgbuf'.
 *
 * Return EX_* error code (from <sysexits.h>) as follows:
 *	EX_OK - successfully retrieved a new image
 *	EX_SOFTWARE - internal software error
 *	EX_IOERR - read error or unparsable binary header
 *	EX_NOINPUT - end-of-file on read
 */
sysex_t
GetImage(int fd)
{
	static int bigendian = 2;	/* set true = big, false = little */
	const char *delim = "\nContent-Length:";
	const size_t delimlen = strlen(delim);
	union {
		unsigned char pbuf[HDRBUFSIZ*2];
		binhdr_gp_t gp;
	} hdr;
	unsigned char *cp_s, *cp, *cp_e;
	ssize_t nr;		/* return value from read(2) */
	size_t imghdrlen;	/* number of image bytes left in hdr.pbuf */
#define	SCANLIMIT	(delimlen + 16)

	if (SCANLIMIT * 2 > HDRBUFSIZ) {
		fprintf(stderr, "%s: internal error: HDRBUFSIZ too small",
			progname);
		return EX_SOFTWARE;
	}

	if (bigendian == 2)	/* initialize endianness of hardware */
		bigendian = (*((char *)(&bigendian)) == 0);

	/*
	 * Initialize start, current, and end pointers for 'hdr.pbuf'
	 * (henceforth referred to as 'pbuf').
	 */
	cp_s = cp = cp_e = hdr.pbuf;

	if (opt_binhdr_gp) {
	    nr = filbuf(fd, &hdr.gp, GP_HDRSIZE);
	    if (opt_debug)
		fprintf(stderr,
			"%s: GP: read %ld/%ld (bom=0x%04x) (bigend:%d)\n",
			progname, (long)nr, (long)GP_HDRSIZE, hdr.gp.bom,
			bigendian);
	    if (hdr.gp.bom != GP_HDR_BOM) {
		fprintf(stderr, "%s: GP: invalid binary header: (bom=0x%04x)\n",
			progname, hdr.gp.bom);
		return EX_IOERR;
	    }
	    imglen = bigendian?
		((hdr.gp.imglen & 0xff) << 8) | ((hdr.gp.imglen & 0xff00) >> 8)
		: hdr.gp.imglen;
	    imghdrlen = 0;
	    if (nr == GP_HDRSIZE)	/* tell error check that all is well */
		nr = HDRBUFSIZ;
	} else {
	    /*
	     * Read one HDRBUFSIZ bytes into 'pbuf'.  On first iteration,
	     * the data is read in at pbuf[0], while on succeeding calls,
	     * the data is read in at pbuf[SCANLIMIT] because the trailing
	     * SCANLIMIT bytes were copied to the head of 'pbuf'.
	     */
	    while ((nr = filbuf(fd, cp_s, HDRBUFSIZ)) == HDRBUFSIZ) {
		/*
		 * Look for 'delim' from pbuf[0] - pbuf[HDRBUFSIZ-SCANLIMIT].
		 * If not found, copy remaining bytes to the start of pbuf[],
		 * read another HDRBUFSIZ bytes, and continue searching...
		 */
		for (cp = hdr.pbuf, cp_e = cp_s + HDRBUFSIZ - SCANLIMIT;
		     cp < cp_e; cp = cp_s + 1) {
			cp_s = cp;
			if (strncasecmp((char *)cp, delim, delimlen) == 0) {
				cp += delimlen;
				/* skip any whitespace to length */
				while (cp < cp_e && isspace(*cp))
					cp++;
				/* length must start with a digit */
				if (cp < cp_e && isdigit(*cp)) {
					/*
					 * Grab the length, then look for
					 * "\r\n\r\n" or "\n\n" to break.
					 * If we dont find it in this buffer,
					 * just give up and start over.
					 */
					imglen = atoi((char *)cp);
					for (; (cp + 4) < cp_e; cp++) {
						if (*(cp) == '\r' &&
						    *(cp + 1) == '\n' &&
						    *(cp + 2) == '\r' &&
						    *(cp + 3) == '\n') {
							cp += 4;
							goto breakout;
						} else if (*(cp) == '\n' &&
						    *(cp + 1) == '\n') {
							cp += 2;
							goto breakout;
						}
					}
				}
			}
		}
breakout:
		if (cp < cp_e)
			break;
		memcpy((void *)hdr.pbuf, (void *)cp, SCANLIMIT);
		cp_s = hdr.pbuf + SCANLIMIT;
	    }

	    /*
	     * The initial bits of the JPEG image start at 'cp' and run
	     * until 'cp_e + SCANLIMIT'.  Copy 'imghdrlen' bytes to the
	     * image buffer now.  Read remaining 'imglen' bytes later.
	     */
	    imghdrlen = (cp_e + SCANLIMIT) - cp;
	    memcpy((void *)imgbuf, (void *)cp, imghdrlen);
	}

	if (nr != HDRBUFSIZ) {
		if (nr < 0) {
			fprintf(stderr, "%s: read: %s\n",
				progname, strerror(errno));
			return EX_IOERR;
		}
		if (opt_debug)
			fprintf(stderr, "%s: read: end of file\n", progname);
		return EX_NOINPUT;
	}

	if (imglen > IMGBUFSIZ) {
		fprintf(stderr, "%s: image too large: %ld bytes (max:%d)\n",
			progname, (long)imglen, IMGBUFSIZ);
		return EX_OK;
	}

	if (opt_debug)
		fprintf(stderr, "%s: image length %ld\n",progname,(long)imglen);

	/* Read (the remaining) image bytes into imgbuf at the proper offset */
	if ((nr = filbuf(fd, imgbuf + imghdrlen, imglen - imghdrlen)) !=
			(imglen - imghdrlen)) {
		if (opt_debug)
			fprintf(stderr,
				"%s: stdin: wanted %ld bytes, got %ld\n",
				progname, (long)(imglen - imghdrlen), (long)nr);
		return EX_NOINPUT;	/* end of file */
	}

	return EX_OK;
}

/*
 * PutImage(cmd, outfil, renfil)
 *
 * Write 'imglen' bytes from image in 'imgbuf'.  If 'cmd' is non-NULL,
 * the image is piped to the command.  If 'outfil' is non-NULL, the
 * image is written to a file (and renamed to 'renfil' if non-NULL).
 *
 * Failures lead to error messages but are never fatal.
 */
void
PutImage(const char *cmd, const char *outfil, const char *renfil)
{
#define	OUTFILNAMSIZ	256
	static char head[OUTFILNAMSIZ*6], tail[OUTFILNAMSIZ*1];
	static int imagecnt = 0;
	char numoutfil[OUTFILNAMSIZ*8];
	const char *flnm = NULL;
	FILE *fp = NULL;

	/*
	 * If '-T' option provided, scan the header looking for a JPEG SOI
	 * and initialize opt_skpcnt to the offset.  It is assumed that the
	 * offset from a particular camera is constant.  If that's not true,
	 * remove the "opt_skipcnt == 0" check below (and send me an email
	 * and I will add yet another command-line option).
	 */
	if (opt_binhdr_tn && opt_skpcnt == 0) {
		const unsigned char *cp, *cp_e;
		for (cp = imgbuf, cp_e = imgbuf + imglen - 1; cp < cp_e; cp++)
			if (*cp == TN_HDR_SOI1 && *(cp + 1) == TN_HDR_SOI2)
				break;
		if (cp == cp_e) {
			if (opt_debug)
				fprintf(stderr, "%s: no JPEG header found\n",
					progname);
			return;
		}
		opt_skpcnt = cp - imgbuf;
		if (opt_debug)
			fprintf(stderr,
				"%s: found JPEG header at offset %d bytes\n",
				progname, opt_skpcnt);
	}

	/*
	 * Ensure that we can skip the prepended header and then do so.
	 */
	if (imglen < opt_skpcnt) {
		if (opt_debug)
			fprintf(stderr,
				"%s: skip count of %d goes past end of image\n",
				progname, opt_skpcnt);
		return;
	}
	imglen -= opt_skpcnt;

	if (opt_debug)
		fprintf(stderr, "%s: WRITE image (%s)\n",
			progname, cmd == NULL? outfil: cmd);

	if (cmd != NULL) {
		if (opt_debug > 2)
			fprintf(stderr, "%s: launch: %s\n", progname, cmd);
		if ((fp = popen(cmd, "w")) == NULL)
			fprintf(stderr, "%s: popen(%s) failed\n",progname,cmd);
	} else if (outfil != NULL) {
		if (opt_numfil) {
			if (imagecnt++ == 0) {
				char *cp;
				strncpy(head, outfil, sizeof(head));
				*tail = head[sizeof(head) - 1] = '\0';
				if ((cp = strrchr(head, '.')) != NULL) {
					strncpy(tail, cp, sizeof(tail));
					tail[sizeof(tail) - 1] = '\0';
					*cp = '\0';
				}
			}
			sprintf(numoutfil, "%s-%08d%s", head, opt_nummax > 0?
				(imagecnt % opt_nummax): imagecnt, tail);
			flnm = numoutfil;
		} else {
			flnm = outfil;
		}
		if (opt_debug > 2)
			fprintf(stderr, "%s: file: %s\n", progname, flnm);
		if ((fp = fopen(flnm, "w")) == NULL)
			fprintf(stderr, "%s: fopen(%s): %s\n",
					progname, flnm, strerror(errno));
	}

	if (fp != NULL) {
		int nw = fwrite(imgbuf + opt_skpcnt, sizeof(char), imglen, fp);
		if (nw != imglen)
			fprintf(stderr, "%s: incomplete write %ld/%ld\n",
				progname, (long)nw, (long)imglen);
	}

	if (cmd != NULL && fp != NULL) {
		if (pclose(fp) == -1)
			fprintf(stderr, "%s: pclose(%s) failed\n",progname,cmd);
	} else if (outfil != NULL && fp != NULL) {
		if (fclose(fp) != 0)
			fprintf(stderr, "%s: fclose(%s): %s\n",
					progname, flnm, strerror(errno));
		if (renfil != NULL) {
			if (rename(flnm, renfil) < 0)
				fprintf(stderr, "%s: rename(%s,%s): %s\n",
					progname, flnm, renfil,
					strerror(errno));
		}
	}
#undef	OUTFILNAMSIZ
}

/*
 * filbuf(fd, buf, siz)
 *
 * Read from STDIN 'siz' bytes from 'fd' into 'buf' and return count.
 *
 * A short count (including 0) is returned only upon an EOF.
 * A -1 is returned on failure, errno is set to indicate read error.
 */
ssize_t
filbuf(int fd, void *buf, size_t siz)
{
	ssize_t nr, totr;
	unsigned char *bp = (unsigned char *)buf;

	totr = 0;
	while (siz > 0) {
		switch (nr = read(fd, bp, siz)) {
		    case -1:
			if (errno == EINTR)
				continue;
			return -1;
		    case 0:
			return totr;
		    default:
			totr += nr;
			siz -= nr;
			bp += nr;
			break;
		}
	}

	return totr;
}

/*
 * GetSock(hoststr, portstr)
 *
 * Initiate a TCP connection to host 'hoststr' on port 'portstr'.
 *
 * Return the file descriptor on success.
 * On failure, an error message is displayed and -1 is returned.
 */
int
GetSock(const char *hoststr, const char *portstr)
{
	const char *errmsgfun = NULL;
	const char *errmsgerr = NULL;
	struct hostent *hp = NULL;
	struct sockaddr_in sin;
	unsigned int addr;
	unsigned short port;
	int fd = -1;

	if (opt_debug)
		fprintf(stderr, "%s: connecting to %s\n", progname, hoststr);

	port = atoi(portstr);
	if ((addr = inet_addr(hoststr)) == INADDR_NONE) {
		if ((hp = gethostbyname(hoststr)) == NULL) {
			errmsgfun = "gethostbyname";
#ifdef	NO_HSTRERROR
			errmsgerr =
				(h_errno == HOST_NOT_FOUND)?
					"Unknown host":
				(h_errno == TRY_AGAIN)?
					"Host name lookup failure":
				(h_errno == NO_RECOVERY)?
					"Unknown server error":
				(h_errno == NO_ADDRESS)?
					"No address associated with name":
					"Unknown resolver error";
#else
			errmsgerr = hstrerror(h_errno);
#endif
			goto errout;
		}
	}

	if ((fd = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		errmsgfun = "socket";
		goto errout;
	}

#ifdef SO_KEEPALIVE
	{
		int on = 1;
		setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE,(char *)&on, sizeof on);
	}
#endif

	/*
	 * When actively connecting to a socket and there are multiple
	 * ip addresses for the target host, pop back to "cagain" for
	 * each new address until we try all of them.
	 */
cagain:
	memset(&sin, 0, sizeof sin);
	if (hp == NULL) {
		sin.sin_family = PF_INET;
		memcpy(&sin.sin_addr, &addr, sizeof addr);
	} else {
		sin.sin_family = hp->h_addrtype;
		memcpy(&sin.sin_addr, hp->h_addr_list[0],
			MIN(hp->h_length, ((int)sizeof sin.sin_addr)));
	}
	sin.sin_port = htons(port);
#ifndef	NO_SINLEN
	sin.sin_len = sizeof (struct sockaddr_in);
#endif

	if (connect(fd, (struct sockaddr *)&sin, sizeof sin) < 0) {
		if (hp != NULL && hp->h_addr_list[1] != NULL) {
			hp->h_addr_list++;
			goto cagain;
		}
		errmsgfun = "connect";
		goto errout;
	}

	return fd;

errout:
	fprintf(stderr, "%s: GetSock(%s,%s): %s: %s\n", progname,
		hoststr, portstr, (errmsgfun == NULL)? "unknown": errmsgfun,
		(errmsgerr == NULL)? strerror(errno): errmsgerr);
	if (fd >= 0)
		close(fd);

	return -1;
}

/*
 * SendURL(fd, pathstr, authstr)
 *
 * Send an HTTP GET request of 'pathstr' to descriptor 'fd'.
 * If 'authstr' is not NULL, include "basic authorization" as well.
 * If debug is enabled, the input stream is read to hopefully
 * get/display an HTTP response.
 *
 * Return EX_* error code (from <sysexits.h>).
 */
sysex_t
SendURL(int fd, const char *pathstr, const char *authstr)
{
	char *cp, sndbuf[8192];
	int len, cc;

	len = snprintf(sndbuf, sizeof(sndbuf),
		"GET /%s HTTP/1.0\r\n%s%s%s\r\n", pathstr,
		authstr == NULL? "": "Authorization: Basic ",
		authstr == NULL? "": authstr,
		authstr == NULL? "": "\r\n");

	/*
	 * Write out the GET request, ensuring that it all goes out.
	 */
	for (cp = sndbuf; (cc = write(fd, cp, len)) != len; ) {
		if (cc < 0) {
			if (errno == EINTR)
				continue;
			else {
				fprintf(stderr,
					"%s: SendURL(/%s): write: %s\n",
					progname, pathstr, strerror(errno));
				return EX_IOERR;
			}
		}
		cp += cc;
		len -= cc;
	}

	/*
	 * If "double debug" mode, try to grab the status line
	 * from the HTTP server and report it.
	 */
	if (opt_debug > 1) {
		if (filbuf(fd, sndbuf, 64) == 64) {
			sndbuf[64] = '\0';
			if ((cp = strchr(sndbuf, '\r')) != NULL)
				*cp = '\0';
			if ((cp = strchr(sndbuf, '\n')) != NULL)
				*cp = '\0';
			fprintf(stderr, "%s: HTTP server reports: %s\n",
				progname, sndbuf);
		}
	}

	return EX_OK;
}


#include <inttypes.h>

/* base64encode() is public domain (http://en.wikipedia.org/wiki/Base64) */
void 
base64encode(const void *data_buf,
	size_t dataLength,
	char *result,
	size_t maxResultLength)
{
	const char	base64chars[] =
	    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	const uint8_t  *data = (const uint8_t *)data_buf;
	size_t		resultIndex = 0;
	size_t		x;
	uint32_t	n = 0;
	int		padCount = dataLength % 3;
	uint8_t		n0, n1, n2, n3;

	/* increment over length of string, three characters at a time */
	for (x = 0; x < dataLength; x += 3) {
		/* these three 8-bit (ASCII) chars become one 24-bit number */
		n = data[x] << 16;
		if ((x + 1) < dataLength)
			n += data[x + 1] << 8;
		if ((x + 2) < dataLength)
			n += data[x + 2];

		/* this 24-bit number gets separated into four 6-bit numbers */
		n0 = (uint8_t) (n >> 18) & 63;
		n1 = (uint8_t) (n >> 12) & 63;
		n2 = (uint8_t) (n >> 6) & 63;
		n3 = (uint8_t) n & 63;

		/* if 1 byte available, then encoding is spread over 2 chars */
		result[resultIndex++] = base64chars[n0];
		if (resultIndex > maxResultLength)
			return;
		result[resultIndex++] = base64chars[n1];
		if (resultIndex > maxResultLength)
			return;

		/* if 2 bytes available, then encoding is spread over 3 chars */
		if ((x + 1) < dataLength)
			result[resultIndex++] = base64chars[n2];
		if (resultIndex > maxResultLength)
			return;

		/* if 3 bytes available, then encoding is spread over 4 chars */
		if ((x + 2) < dataLength)
			result[resultIndex++] = base64chars[n3];
		if (resultIndex > maxResultLength)
			return;
	}

	/*
	 * create and add padding that is required if we did not have a
	 * multiple of 3 number of characters available
	 */
	if (padCount > 0) {
		for (; padCount < 3; padCount++) {
			result[resultIndex++] = '=';
			if (resultIndex > maxResultLength)
				return;
		}
	}
	result[resultIndex] = 0;
}
